#
# This is a PowerShell Script written by KC1IUC to download the Static Daily data dumps
# from DMR-MARC and format them to the "CSV" import method the Anytone AT-D868UV and TYT MD-UV380 / MD-UV390 CPS Uses
# Last Updated 2019-03-11-00:00-EST
# Contact Info: erik@KC1IUC.com
# For Updates please check KC1IUC.com https://www.kc1iuc.com/downloads/RadioDL.zip
#
# See bottom for changelog
#
# This script REQUIRES Windows PowerShell 5.1 or greater. This is included in most newer versions of Windows 10 (From Windows 10 Anniversary Update 1607 from 2016 and up) but not in Windows 7 or Windows 8.1. 
# Windows 7 and 8.1 can download the Windows Management Framework 5.1 update to install PowerShell 5.1 here: https://www.microsoft.com/en-us/download/details.aspx?id=54616
# Windows 10 users on a update older than 1607 can update to the latest build with this tool, currently build 1903.  https://www.microsoft.com/en-us/software-download/windows10
# Having .Net 4.7.2 may also be a good idea. If you Have Windows 10 1803 or newer, it's not needed, but any version of Windows older than that can get it from here https://dotnet.microsoft.com/download/dotnet-framework/net472
# 
$DoTYT = $true #Set True/False for TYT output
$DoAnytone = $true #Set True/False for Anytone Output
$ShortenNames = $false #Set this to $true or $false to do Country and State abbreviations. Setting to YES adds a lot of CPU time to process. 
#Enter the URL of the DMR-MARC static Users CSV here. This should not normally change. You can do HTTPS or HTTP. Non-quoted or quoted work, non-quited is prefered as it is a smaller size.
#$DMRMARCURL = "https://www.radioid.net/static/users.csv"
$DMRMARCURL = "https://www.radioid.net/api/dmr/user/?id="
$HamDigitalURL = "https://ham-digital.org/status/users.csv"
$DLMethod = "BITS" # Set this to BITS or WebClient (Bits is better, but has issues with some HTTPS Self Signed Certs, so use BITS and only change if needed)
#DO NOT EDIT AFTER THIS LINE
#These are arrays to replace Country and States with abbreviations
$PSversion = $PSVersionTable.PSVersion
if ($PSVersion -lt 5.1) {
	"Your Windows PowerShell Version is less than 5.1."
	"You need to update to PowerShell newer than 5.1 to run this correctly."
	"If you are running Windows 7, or 8.1 please download Windows Management Framework 5.1 from this link"
	"https://www.microsoft.com/en-us/download/details.aspx?id=54616"
	"If you are running Windows 10, please update to a Windows 10 Anniversary Update 1607 or later"
	"You can download the Windows 10 Upgrade Assistant app from this link"
	"https://www.microsoft.com/en-us/software-download/windows10"
	"You may also wish to make sure you have the latest .Net Framework 4.7.2 Installed. If you do not get it from this link"
	"https://dotnet.microsoft.com/download/dotnet-framework/net472"
	"This script will now exit. Please try again after installing the patches and rebooting."
	Read-Host -Prompt "Press Enter to exit."
	}
else {
"PowerShell Version Greater than 5.1, proceeding..."
$NationShort = @{
  "United States" = "USA";
  "United Kingdom" = "UK"
}
$StateShort = @{
	"Alberta" = "AB";
	"Alaska" = "AK";
	"Alabama" = "AL";
	"Arkansas" = "AR";
	"American Samoa" = "AS";
	"Arizona" = "AZ";
	"British Columbia" = "BC";
	"California" = "CA";
	"Canada" = "CD";
	"Colorado" = "CO";
	"Connecticut" = "CT";
	"Dist. Of Columbia" = "DC";
	"Delaware" = "DE";
	"Foreign Countries" = "FF";
	"Florida" = "FL";
	"Georgia" = "GA";
	"Guam" = "GU";
	"Hawaii" = "HI";
	"Iowa" = "IA";
	"Idaho" = "ID";
	"Illinois" = "IL";
	"Indiana" = "IN";
	"Kansas" = "KS";
	"Kentucky" = "KY";
	"Louisiana" = "LA";
	"Massachusetts" = "MA";
	"Manitoba" = "MB";
	"Maryland" = "MD";
	"Maine" = "ME";
	"Michigan" = "MI";
	"Minnesota" = "MN";
	"Missouri" = "MO";
	"Mississippi" = "MS";
	"Montana" = "MT";
	"Mexico" = "MX";
	"New Brunswick" = "NB";
	"North Carolina" = "NC";
	"North Dakota" = "ND";
	"Nebraska" = "NE";
	"New Hampshire" = "NH";
	"New Jersey" = "NJ";
	"Newfoundland and Labrador" = "NL";
	"New Mexico" = "NM";
	"Nova Scotia" = "NS";
	"Northwest Territories" = "NT";
	"Nunavut" = "NU";
	"Nevada" = "NV";
	"New York" = "NY";
	"Ohio" = "OH";
	"Oklahoma" = "OK";
	"Ontario" = "ON";
	"Oregon" = "OR";
	"Pennsylvania" = "PA";
	"Prince Edward Island" = "PE";
	"Puerto Rico" = "PR";
	"Quebec" = "QC";
	"Rhode Island" = "RI";
	"South Carolina" = "SC";
	"South Dakota" = "SD";
	"Saskatchewan" = "SK";
	"Tennessee" = "TN";
	"Trust Territory" = "TT";
	"Texas" = "TX";
	"Unknown" = "UN";
	"Utah" = "UT";
	"Virginia" = "VA";
	"Virgin Islands" = "VI";
	"Vermont" = "VT";
	"Washington" = "WA";
	"Wisconsin" = "WI";
	"West Virginia" = "WV";
	"Wyoming" = "WY";
	"Yukon" = "YT"
}
$MyPath = Split-Path -Parent $PSCommandPath #Gets the scripts absolute path and removes the file name from it to give the working directory
$TimeOfDL = Get-Date -UFormat "%Y%m%d" #Gets Date in YYYYMMDD format.
$FileHourMinSec = Get-Date -UFormat "%H%M%S" #Gets Time in HHMMSS format.
#Sets 7z.exe Location depending on 32 or 64bit
if ([Environment]::Is64BitProcess -ne [Environment]::Is64BitOperatingSystem)
{ $7zexe = "$MyPath\System\7za32.exe"}
Else
{ $7zexe = "$MyPath\System\7za64.exe" }
$OutPutFilePath = "$MyPath\Users.Anytone.$TimeOfDL.$FileHourMinSec.csv" #Sets the absolute output file name for Anytone
$OutPutFilePathTYT = "$MyPath\Users.TYT.$TimeOfDL.$FileHourMinSec.csv" #Sets the absolute output file name for TYT
#Downloads the file via BITS if the source is HTTP, and WebClient if it is HTTPS since BITS is better but has issues with invalid and self signed SSL Certificates and we can bypass that with WebClient. 
if ($DLMethod -like '*WebClient*') {
	"Downloading the daily files via WebClient. Please be patient, HamDigital is a little slow at times."
	[System.Net.ServicePointManager]::ServerCertificateValidationCallback = { $true }
	[Net.ServicePointManager]::SecurityProtocol = [Net.SecurityProtocolType]::Tls12
	$webClient = new-object System.Net.WebClient
	$webClient.DownloadFile( "$DMRMARCURL", "$MyPath\RadioIDusers.json" )
	$webClient.DownloadFile( "$HamDigitalURL", "$MyPath\HamDigitalusers.csv" )
	}	Else {
	Import-Module BitsTransfer
	Start-BitsTransfer -Source "$DMRMARCURL","$HamDigitalURL" -Destination "$MyPath\RadioIDusers.json","$MyPath\HamDigitalusers.csv" -Priority Foreground -DisplayName "Download_User_Lists" -Description "Downloading RadioID and HamDigital Daily User List via BITS. Please be patient, HamDigital is a little slow at times."
	}
#This Checks to see if the script was run before. If it was, it checks the last known hash against the files we just downloaded and exits if it is the same data.
Clear
"Checking the File Hash for RadioID against the last run file hash"
if ((Test-Path "$MyPath\System\LastRun.txt") -And (Test-Path "$MyPath\System\LastRunHD.txt"))  {
	Get-FileHash "$MyPath\RadioIDusers.json" | Select-Object -ExpandProperty Hash | Out-File -Encoding UTF8 "$MyPath\System\NewHash.txt"
	Get-FileHash "$MyPath\HamDigitalusers.csv" | Select-Object -ExpandProperty Hash | Out-File -Encoding UTF8 "$MyPath\System\NewHashHD.txt"
	$newhash = gc -Encoding UTF8 "$MyPath\System\NewHash.txt"
	$newhashHD = gc -Encoding UTF8 "$MyPath\System\NewHashHD.txt"
	$oldhash = gc -Encoding UTF8 "$MyPath\System\LastRun.txt"
	$oldhashHD = gc -Encoding UTF8 "$MyPath\System\LastRunHD.txt"
	If ((Compare-Object $oldhash $newhash) -Or (Compare-Object $oldhashHD $newhashHD))
	{
		Clear
		"New data! Proceeding with formatting."
		#Removing Old Hash, Saving New Hash
		Remove-Item "$MyPath\System\LastRun.txt", "$MyPath\System\LastRunHD.txt"
		Rename-Item  -Path "$MyPath\System\NewHash.txt" -NewName "$MyPath\System\LastRun.txt"
		Rename-Item  -Path "$MyPath\System\NewHashHD.txt" -NewName "$MyPath\System\LastRunHD.txt"
	}
	else
	{
		Clear
		"The files are the same. You already have the latest datafile."
		Remove-Item "$MyPath\System\LastRun.txt", "$MyPath\System\LastRunHD.txt"
		Rename-Item  -Path "$MyPath\System\NewHash.txt" -NewName "$MyPath\System\LastRun.txt"
        Rename-Item  -Path "$MyPath\System\NewHashHD.txt" -NewName "$MyPath\System\LastRunHD.txt"
		Write-host "Would you like to override and format the files anyway? (Default is No and program will exit)" -ForegroundColor Yellow 
		$Readhost = Read-Host " ( y / n ) " 
		Switch ($ReadHost) 
			{ 
				Y {$OverrideDownload=$true} 
				N {$OverrideDownload=$false} 
				Default {$OverrideDownload=$false} 
			}
		If ($OverrideDownload -eq $false)
			{
				Remove-Item "$MyPath\RadioIDusers.json","$MyPath\HamDigitalusers.csv"
				Exit
			}		
	}
	}
Else
	{
    "We haven't run this before, getting hashes and moving on!"
	Get-FileHash "$MyPath\RadioIDusers.json" | Select-Object -ExpandProperty Hash | Out-File -Encoding UTF8 "$MyPath\System\LastRun.txt"
    Get-FileHash "$MyPath\HamDigitalusers.csv" | Select-Object -ExpandProperty Hash | Out-File -Encoding UTF8 "$MyPath\System\LastRunHD.txt"
	}
Clear
#Archiving Old CSV files to a 7zip file
"Now archiving old files to Archive 7Zip file"
Get-ChildItem $MyPath | where {$_.Name -like "Users.*.csv"} | ForEach-Object { & "$7zexe" -sdel -t7z -mx9 a "$MyPath\Archive.7z" $_.Name } | Out-Null
# This section converts the JSON output from RadioID to a CSV. They do offer a direct CSV output, however we have to resort and format the data order anyway so we'll just use JSON. We can change this later on maybe. 
clear
"Now converting RadioID JSON data to CSV. This is CPU intenseive and may take a few minutes."
((gc -Path "$MyPath\RadioIDusers.json" -Encoding UTF8) | ConvertFrom-Json).results | Select @{n="Radio ID";e={$_.id}},@{n="Callsign";e={$_.callsign}},@{n="Name";e={$_.fname + " " + $_.surname}},@{n="City";e={$_.city}},@{n="State";e={$_.state}},@{n="Country";e={$_.country}},@{n="Remarks";e={$_.remarks}} | Export-CSV -NoTypeInformation -Encoding UTF8 -Path "$MyPath\RadioIDusers.csv"
# This Combines the RadioID and HamDigital Databases. It pushes a header and two files together, then sorts by ID, then State. Then sorts by ID again with unique flag that throws duplicated info out, which should be HamDigital Data that has missing State column info. Also replaces set Countries and States with abbreviations #####
#### Start Combination of files ####
Clear
if ($ShortenNames -eq $true) 
	{
	"Now Importing and Sorting RadioID and HamDigital User Contact Lists, and deduplicating IDs. Also shortening Country and State names. This is VERY CPU intensive, Please be patient. It can take 5+ minutes."
	Import-CSV -Encoding UTF8 "$MyPath\RadioIDusers.csv","$MyPath\HamDigitalusers.csv" -Delimiter "," -Header "Radio ID","Callsign","Name","City","StateOld","CountryOld","Remarks" | Select-Object *, @{ Name = "Country"; Expression = { $NationShort[$_.CountryOld] } } | ForEach-Object {If  ([string]::IsNullOrEmpty($_.Country)) {$_.Country = $_.CountryOld}$_} | Select-Object *, @{ Name = "State"; Expression = { $StateShort[$_.StateOld] } } | ForEach-Object {If  ([string]::IsNullOrEmpty($_.State)) {$_.State = $_.StateOld}$_} | Select 'Radio ID','Callsign','Name','City','State','Country','Remarks' | Sort 'Radio ID','State' | Sort 'Radio ID' -Unique | Export-CSV -NoTypeInformation -Encoding UTF8 -Path "$MyPath\usersraw.csv"
	}
if ($ShortenNames -eq $false)
{
	"Now Importing and Sorting RadioID and HamDigital User Contact Lists, and deduplicating IDs. This is CPU intensive, Please be patient. It can take a few minutes."
	Import-CSV -Encoding UTF8 "$MyPath\RadioIDusers.csv","$MyPath\HamDigitalusers.csv" -Delimiter "," -Header "Radio ID","Callsign","Name","City","State","Country","Remarks" | Select 'Radio ID','Callsign','Name','City','State','Country','Remarks' | Sort 'Radio ID','State' | Sort 'Radio ID' -Unique | Select-Object -SkipLast 1 | Export-CSV -NoTypeInformation -Encoding UTF8 -Delimiter '|' -Path "$MyPath\usersraw.csv"
	}
Clear
#### End Combination of files ####
#
If ($DoAnytone -eq $true)
{
	#### Anytone Formatting Start ####
	Clear
	"Now formatting the Anytone AT-D868UV CSV file"
	"`"No.`",`"Radio ID`",`"Callsign`",`"Name`",`"City`",`"State`",`"Country`",`"Remarks`",`"Call Type`",`"Call Alert`"" | Out-File -Encoding UTF8 -FilePath "$OutPutFilePath" #Creates the header that Anytone wants
	Start-Sleep -Milliseconds 2000
	Add-Content -Path "$OutPutFilePath" -Value (gc -Encoding UTF8 "$MyPath\usersraw.csv" | Select-Object -Skip 1 | % {$_.Replace(',','')} |% {$_.Replace('|',',')} | %{$i++;"""$($i)"",$_"} | foreach {$_ + ",`"Private Call`",`"None`"" } ) #This adds the combined file to the header, skipping the first line that had old header info, and prefixing each line with a number and adding private call info to the end of each line skipping the first header line.
	##### Anytone Formatting End ####
	}
	#
If ($DoTYT -eq $true)
	{
	#### TYT Formatting Start ####
	Clear
	"Now formatting the TYT MD-UV380 / MD-UV390 CSV file"
	Import-CSV -Delimiter '|' -Encoding UTF8 -Header "Radio ID","Callsign","Name","City","State","Country","Remarks","NickName" -Path "$MyPath\usersraw.csv" | Select "Radio ID","Callsign","Name","Nickname","City","State","Country" | where{$_.Country -ne "China"} | where{$_.Country -ne ""} | where{$_.Country -ne "Brazil"} | where{$_.Country -ne "Spain"} | where{$_.Callsign -ne "radioid"} | Select -Skip 1 | Export-CSV -Encoding UTF8 -NoTypeInformation -Path "$MyPath\TYTTemp.csv" -Delimiter '|'
	gc -Encoding UTF8 -Path "$MyPath\TYTTemp.csv" | % {$_.Replace('"','')} | % {$_.Replace(',','')} |% {$_.Replace('|',',')} | foreach {$_ + ",,,,,," } | Out-File -Encoding UTF8 -FilePath "$MyPath\TYTUTF8BOM.csv"#This removes quotes from the CSV file since thats how TYT wants it and removed any commas then replaces |'s with , that we used for deliminator
	$TYTFile = gc -Encoding UTF8 "$MyPath\TYTUTF8BOM.csv" #This line and next two convert the CSV to strict UTF8 because apparently TYT does NOT like UTF8-BOM encoding that PowerShell outputs with its UTF8 setting
	$Utf8NoooBomEncoding = New-Object System.Text.UTF8Encoding $False
	[System.IO.File]::WriteAllLines($OutPutFilePathTYT, $TYTFile, $Utf8NoooBomEncoding)
	Remove-Item "$MyPath\TYTTemp.csv","$MyPath\TYTUTF8BOM.csv"
	#### TYT Formatting End ####
	}
#
#Finding out the number of records.
$NewLines = gc -Encoding UTF8 "$MyPath\usersraw.csv" | Measure-Object -Line | Select-Object -ExpandProperty Lines
$NewLines | Out-File -Encoding UTF8 "$MyPath\System\NewLines.txt"
Clear
"Cleaning up..."
#Removes a few temp files we made
Remove-Item "$MyPath\usersraw.csv", "$MyPath\RadioIDusers.csv","$MyPath\RadioIDusers.json","$MyPath\HamDigitalusers.csv"
Clear
#If we ran before show last run record lines, or if its fresh run just this runs record count
If (Test-Path "$MyPath\System\OldLines.txt")
	{
	$OldLines = gc -Encoding UTF8 "$MyPath\System\OldLines.txt"
    $LinesChanged = $NewLines - $OldLines
	"There are now $NewLines records. Last run there were $OldLines records."
    If ($LinesChanged -ge 0)
    {
    "$LinesChanged records have been added."
    }
    Else
    {
    $LinesChangedNeg = 0 - $LinesChanged
    "$LinesChangedNeg records have been subtracted."
    }
	Remove-Item "$MyPath\System\OldLines.txt"
	}
Else
	{
	"There are now $NewLines records."
	}
Rename-Item -Path "$MyPath\System\NewLines.txt" -NewName "$MyPath\System\OldLines.txt"
"Your CSV files are ready for import into your CPS."
"Anytone AT-D868UV CSV file is located at:"
"$OutPutFilePath"
"TYT MD-UV380 / MD-UV390 CSV file is located at:"
"$OutPutFilePathTYT"
Read-Host -Prompt "Press Enter to exit."
}
# Changelog
# 2018-05-21-15:54-EST - Added TYT MD-UV380 / MD-UV390 formatting options and removed some unnecisary comment lines.
# 2018-05-21-20:16-EST - Added some things to convert the TYT output to strict UTF8 as PowerShell outputs UTF8-BOM which TYT apparently does NOT find acceptable.
# 2018-05-25-16:52-EST - Added entire routing to download file from HamDigital and merge just the European 2xxxxxx IDs into the RadioID database, sort, and save. In the processes rewrote Anytone formatting to use Export-Csv instead of just string maniplation.
# 2018-06-10-15:53-EST - Added logic to check for new file hashes / changes for the HamDigital user list. 
# 2018-06-11-19:27-EST - Rewrote comibining code. Instead of just selecting the 2xxxxxx ID's from HamDigital, it now combines the two files and sorts it and only keeps the duplicate lines that have information (or more info) in the State field since HamDigital leaves State field blank. It also does it much more efficiently. Also rewrote Anytone formatting to be MUCH more efficient and do everything in one line basically. Adjusted TYT formatting to except quoted input, less work this way. We still download non-quoted input from sources as the files are smaller, and Import-CSV doesn't seem to care. Took out prefix-name option, why bother. Also forced BITS by default, is now user selectable instead of going off http/https in URL. 
# 2018-09-26-21:51-EST - Not sure can't remember what I changed. 
# 2019-02-07-13:37-EST - Changed United Kingdom, United States to shorted to UK and USA. Also changed states to abbreviations. This should cut down on file size a little bit. 
# 2019-02-28-17:09-EST - CHANGED A LOT. Made the shortening of Country and States and optional toggle. I also changed RadioID data to use JSON output and convert it over to CSV. This adds a bit more CPU time, but worth it. Also removed some unnecisary UTF8-BOM conversion prior to TYT import. 
# 2019-03-01-11:00-EST - Changed TYT formatting to use Import-CSV instead of combining text files. Also had TYT remove a few countries (China and Brazil) as I found out it has 120,000 contact limit. This can be modified on the fly or other countries added or removed to make it fit. We're at close to 124,000 contacts now so what we take out will need to grow. Also added a Toggle for both TYT and Anytone formatting. 
# 2019-03-05-1513-EST - Fixed a typo
# 2019-05-24-1331-EST - Added logic to detect when PowerShell is less than the latest version, 5.1, as apparently version 5.1 or greater is requeired by some functions I use. Added links and prompts for users to download PowerShell 5.1. It's been on all Windows 10 versions since 1607 in 2016, which is why I never had an issue with it since I wrote this AFTER those updates. 